/**********************************************************************

SerialCommand.cpp
COPYRIGHT (c) 2013-2016 Gregg E. Berman

Part of DCC++ BASE STATION for the Arduino

**********************************************************************/

// DCC++ BASE STATION COMMUNICATES VIA THE SERIAL PORT USING SINGLE-CHARACTER TEXT COMMANDS
// WITH OPTIONAL PARAMTERS, AND BRACKETED BY < AND > SYMBOLS.  SPACES BETWEEN PARAMETERS
// ARE REQUIRED.  SPACES ANYWHERE ELSE ARE IGNORED.  A SPACE BETWEEN THE SINGLE-CHARACTER
// COMMAND AND THE FIRST PARAMETER IS ALSO NOT REQUIRED.

// See SerialCommand::parse() below for defined text commands.

#include "SerialCommand.h"
#include "DCCpp_Uno.h"
#include "Accessories.h"
#include "Sensor.h"
#include "Outputs.h"
#include "EEStore.h"
#include "Comm.h"
#include <LiquidCrystal.h>
LiquidCrystal lcd(8, 9, 4, 2, 6, 7);

extern int __heap_start, *__brkval;

/////////////////////////////////////////////////////////////////////////////////////////////////////////           
//////////////////////////////////////  CJR Declarations  ///////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////   

int SW1_4_AN_VAL = 1023; //SW1-4 From LCD_SHIELD
int SW56_AN_VAL = 1023; //SW5,6 Formerly HORN_BELL
int SW78_AN_VAL = 1023; //SW7,8 Formerly THR_MULT
int SW910_AN_VAL = 1023; //SW9,10 Formerly MUTE_SW

int setpoint[LOCOS] = {0,0,0,0,0,0,0,0,0,0,0,0}; //variable to tell the DCC the throttle setting per loco, index=LOCO_NUM-1
int setpointold[LOCOS] = {0,0,0,0,0,0,0,0,0,0,0,0};
byte dccmanuf[LOCOS] = {1,1,0,0,0,0,0,0,0,2,0,0};  //0 = generic no sound, 1 = Paragon2, 2 = LokSound Select
byte FWD_REV_TOGGLE[LOCOS] = {1,1,1,1,1,1,1,1,1,1,1,1};
byte F0_F4[LOCOS] = {144,144,144,144,144,144,144,144,144,144,144,144}; //F0 on for heeadlights on.
byte F0_F4_OLD[LOCOS] = {144,144,144,144,144,144,144,144,144,144,144,144}; 
byte F5_F8[LOCOS] = {176,176,176,176,176,176,176,176,176,176,176,176}; 
byte F5_F8_OLD[LOCOS] = {176,176,176,176,176,176,176,176,176,177,176,176};
byte F9_F12 = 160; 
byte F9_F12_OLD = 160;
byte F13_F20 = 0;
byte F13_F20_OLD = 0;
byte F1_TOGGLE[LOCOS] = {0,0,0,0,0,0,0,0,0,0,0,0}; //formerly BELL_TOGGLE
byte F8_TOGGLE[LOCOS] = {0,0,0,0,0,0,0,0,0,0,0,0}; //formerly MUTE_SW_TOGGLE
byte THROTTLE_UP_PB = 0;
byte THROTTLE_DN_PB = 0;
byte TRK_PWR_TOGGLE_PB = 0;
byte THR_MULT_PB = 1;
byte LOCO_NUM = 1;
//SW1-SW4 are from the LCD shield, 
byte SW2 = 0; //F9,F10, 
byte SW5 = 0; //F2, F3 (horn/whistle and couplers): 10K pulldown on A5
byte SW6 = 0; //F1 (bell): 4.7K pulldown on A5
byte SW7 = 0; //select, shift key: 10K pulldown on A4
//SW8 fine speed control select: 4.7K pulldown on A4
byte SW9 = 0; //F13,F14 (more sound effects): 10K pulldown on A3
byte SW10 = 0; //combined mute/vol PB: 4.7K pulldown on A3
byte PROG_MODE = 0;

/////////////////////////////////////////////////////////////////////////////////////////////////////////           
//////////////////////////////////////  END CJR Declarations  ///////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////


char SerialCommand::commandString[MAX_COMMAND_LENGTH+1];
volatile RegisterList *SerialCommand::mRegs;
volatile RegisterList *SerialCommand::pRegs;
CurrentMonitor *SerialCommand::mMonitor;


///////////////////////////////////////////////////////////////////////////////

void SerialCommand::init(volatile RegisterList *_mRegs, volatile RegisterList *_pRegs, CurrentMonitor *_mMonitor){
  mRegs=_mRegs;
  pRegs=_pRegs;
  mMonitor=_mMonitor;
  sprintf(commandString,"");
  lcd.begin(16, 2);
  lcd.print("I Like Trains"); //or any other text you want
  lcd.setCursor(0, 1);
  lcd.print ("Code Version MR1"); //increment with code changes to keep track of what is programmed
  
  delay (3000);
  lcd.setCursor(0, 0);
  lcd.print ("                ");
  lcd.setCursor(0, 1);
  lcd.print ("                ");
  lcd.setCursor(0, 0);
  lcd.print ("Locomotive: ");
  lcd.setCursor (12, 0);
  lcd.print ("  ");
  lcd.setCursor (12, 0);
  lcd.print (LOCO_NUM);
} // SerialCommand:SerialCommand

///////////////////////////////////////////////////////////////////////////////

void SerialCommand::process(){
  char c;
    
  #if COMM_TYPE == 0
  
    while(INTERFACE.available()>0){    // while there is data on the serial line
    c=INTERFACE.read();
	if(c=='<')                    // start of new command
    sprintf(commandString,"");
	else if(c=='>'){               // end of new command
    parse(commandString); 
    Serial.println(commandString);
		}
  else if(strlen(commandString)<MAX_COMMAND_LENGTH)    // if comandString still has space, append character just read from serial line
    sprintf(commandString,"%s%c",commandString,c);     // otherwise, character is ignored (but continue to look for '<' or '>')
    }
 
/////////////////////////////////////////////////////////////////////////////////////////////////////////           
/////////////////////////////////////////////  CJR MAIN MODS  ///////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////      
	
    SW1_4_AN_VAL = analogRead(SW1_4);  //Reading 5 swithces on LCD SHIELD
    if (SW1_4_AN_VAL > 80 and SW1_4_AN_VAL < 200) THROTTLE_UP_PB = 1;  
    else THROTTLE_UP_PB = 0;
    if (SW1_4_AN_VAL > 210 and SW1_4_AN_VAL < 400) THROTTLE_DN_PB = 1;
    else THROTTLE_DN_PB = 0;

/////////////////////////////////////////////////////////////////////////////////////////////////////////           
/////////////////////////////  Toggles track power off and on  //////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////     

    if (SW1_4_AN_VAL > 610 and SW1_4_AN_VAL < 850 and PROG_MODE == 0) {  //shut off and ignore main track power toggle when in program mode.
      TRK_PWR_TOGGLE_PB = !TRK_PWR_TOGGLE_PB;
      delay (50000);
      for (int n=0; n<LOCOS; n++) { //resets all locos to forward, speed = 0 whenever TRK_PWR_TOGGLE_PB changes state
        setpoint[n] = 0;
        FWD_REV_TOGGLE[n] = 1;
        sprintf(commandString,"t %d %d %d %d",(n+1),(n+1),setpoint[(n)], FWD_REV_TOGGLE[(n)]);
        parse(commandString);
        Serial.print("<");
        Serial.print(commandString);
        Serial.println(">");
        setpointold[n] = 0;
      }
	if (FWD_REV_TOGGLE[(LOCO_NUM-1)] == 0) { //REVERSE
	  lcd.setCursor(0, 1);
	  lcd.print ("REV");
	  } 
    else if (FWD_REV_TOGGLE[(LOCO_NUM-1)] == 1) { //FORWARD
	  lcd.setCursor(0, 1);
	  lcd.print ("FWD");
	  }
    if (TRK_PWR_TOGGLE_PB == 0) {
		parse("0"); //to turn off track power
		Serial.print("<");
		Serial.print(0);
		Serial.println(">");
        lcd.setCursor(13, 1);  
        lcd.print("OFF");
        lcd.setCursor(8, 1);  
        lcd.print("   "); //when track power is turned off, blank out speed on LCD.
      }
      else if (TRK_PWR_TOGGLE_PB == 1) {
        parse("1"); //to turn on track power
        Serial.print("<");
        Serial.print(1);
        Serial.println(">");
        lcd.setCursor(13, 1);  
        lcd.print("ON ");
        lcd.setCursor(0, 0);
        lcd.print ("Locomotive: ");
        lcd.setCursor (12, 0);
        lcd.print ("  ");
        lcd.setCursor (12, 0);
        lcd.print (LOCO_NUM);
      }
    }
/////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////  Locomotive Select by holding Select/Shift and Throttle up/down  /////////////////////////   
/////////////////////////////////////////////////////////////////////////////////////////////////////////           

	SW78_AN_VAL = analogRead(SW78);  //Reading SW7,8 on A4 
	if (SW78_AN_VAL > 601 and SW78_AN_VAL < 800) SW7 = 1;
    else SW7 = 0;
    if (SW7 == 1 and PROG_MODE == 0 and THROTTLE_UP_PB == 1 ) { // SW7, 10K for "SELECT" pushed, lets you select loco using throttle up/down while held
      LOCO_NUM = (LOCO_NUM + 1);
      delay (50000);
      if (LOCO_NUM ==(LOCOS+1)) LOCO_NUM = 1;
      lcd.setCursor (12, 0);
      lcd.print ("  ");
      lcd.setCursor(12, 0);
      lcd.print (LOCO_NUM);
      lcd.setCursor(4, 1);
      lcd.print ("SPD");
      if (FWD_REV_TOGGLE[(LOCO_NUM-1)] == 0) { //REVERSE
        lcd.setCursor(0, 1);
        lcd.print ("REV");
        }
      else if (FWD_REV_TOGGLE[(LOCO_NUM-1)] == 1) { //FORWARD
        lcd.setCursor(0, 1);
        lcd.print ("FWD");
        }
      lcd.setCursor(8, 1);
      lcd.print ("   ");
      lcd.setCursor(8, 1);  
      lcd.print(setpoint[(LOCO_NUM-1)]);
    }
    else if (SW7== 1 and PROG_MODE == 0 and THROTTLE_DN_PB == 1 ) { 
      LOCO_NUM = (LOCO_NUM -1);
      delay (50000);
      if (LOCO_NUM ==0) LOCO_NUM = LOCOS;
      lcd.setCursor (12, 0);
      lcd.print ("  ");
      lcd.setCursor(12, 0);
      lcd.print (LOCO_NUM);
      lcd.setCursor(4, 1);
      lcd.print ("SPD");
      if (FWD_REV_TOGGLE[(LOCO_NUM-1)] == 0) { //REVERSE
        lcd.setCursor(0, 1);
        lcd.print ("REV");
        }
      else if (FWD_REV_TOGGLE[(LOCO_NUM-1)] == 1) { //FORWARD
        lcd.setCursor(0, 1);
        lcd.print ("FWD");
        }
      lcd.setCursor(8, 1);
      lcd.print ("   ");
      lcd.setCursor(8, 1);  
      lcd.print(setpoint[(LOCO_NUM-1)]);
    }
 
/////////////////////////////////////////////////////////////////////////////////////////////////////////              
///////////////////////////////////////  Toggles direction  /////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////              

    if (SW1_4_AN_VAL < 70 and TRK_PWR_TOGGLE_PB == 1) {
      FWD_REV_TOGGLE[(LOCO_NUM-1)] = !FWD_REV_TOGGLE[(LOCO_NUM-1)];
      delay (50000);
      setpoint[(LOCO_NUM-1)] = 0;
      sprintf(commandString,"t %d %d %d %d",LOCO_NUM,LOCO_NUM,setpoint[(LOCO_NUM-1)],FWD_REV_TOGGLE[(LOCO_NUM-1)]);
      parse(commandString);
      Serial.print("<");
      Serial.print(commandString);
      Serial.println(">");
      if (FWD_REV_TOGGLE[(LOCO_NUM-1)] == 0) { //REVERSE
        lcd.setCursor(0, 1);
        lcd.print ("REV");
        }
      else if (FWD_REV_TOGGLE[(LOCO_NUM-1)] == 1) { //FORWARD
        lcd.setCursor(0, 1);
        lcd.print ("FWD");
        }
    }
/////////////////////////////////////////////////////////////////////////////////////////////////////////              
//////////////////  F1, F2, F3 Functions, Horn, Bell, and Coupler Clank  ////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////              

    SW56_AN_VAL = analogRead(SW56);  
    if (SW56_AN_VAL > 451 and SW56_AN_VAL < 600) SW6 = 1; //only 4.7K SW6 (BELL/F1) pushed
	  else SW6 = 0;
    if (SW56_AN_VAL > 601 and SW56_AN_VAL < 800) SW5 = 1; //only 10K SW5 (HORN/F2) pushed
	  else  SW5 = 0;
    if (SW56_AN_VAL < 450) {  //both switches pushed
      SW5 = 1;
      SW6 = 1;
    }
    if (SW6 ==1) {
      F1_TOGGLE[(LOCO_NUM-1)] = !F1_TOGGLE[(LOCO_NUM-1)];
      delay (50000);
    }
    if (F1_TOGGLE[(LOCO_NUM-1)] ==1 and SW5==1 and SW7==0) F0_F4[(LOCO_NUM-1)] = 147; //Do both horn and bell if both are pushed. F1+F2
    //SW7 is used as a shift or select key to double up functions on buttons if pushed simultaneously.
	  else if (F1_TOGGLE[(LOCO_NUM-1)] ==1 and SW5 ==1 and SW7 == 1) F0_F4[(LOCO_NUM-1)] = 149; //Do both coupler clank and bell if both are pushed. F1+F2    
	  else if (SW5==1 and SW7==0) F0_F4[(LOCO_NUM-1)] = 146; //blow horn while SW5 (HORN PB) is pushed (F2)
	  else if (SW5==1 and SW7==1) F0_F4[(LOCO_NUM-1)] = 148; //Clank Coupler when SW7 and SW5 pushed (F3)
	  else if (F1_TOGGLE[(LOCO_NUM-1)] ==1) F0_F4[(LOCO_NUM-1)] = 145; //ring bell while BELL_TOGGLE == 1, toggled by PB (F1)
	  else if (F1_TOGGLE[(LOCO_NUM-1)] ==0 and SW5 ==0) F0_F4[(LOCO_NUM-1)] = 144; 
		//writing 128 sets F0-F4 = 0
		//writing 144 sets F0 = 1, F1-F4 = 0 (horn and bell off)
		//writing 145 sets F0, F1 = 1 and F2, F3, F4 = 0 (bell on)
		//writing 146 sets F0, F2 = 1 and F1, F3, F4 = 0 (horn on)
		//writing 147 sets F0, F1, F2 = 1 and F3, F4 = 0 (horn and bell on)
		//writing 148 sets F0, F3 = 1, and F1, F2, F4 = 0 (coupler clank)
		//Writing 149 sets F0, F1, F3 = 1 and F2, F4 = 0 (coupler clank while bell on)

    if (F0_F4[(LOCO_NUM-1)] != F0_F4_OLD[(LOCO_NUM-1)]) {
		sprintf(commandString,"f %d %d",LOCO_NUM,F0_F4[(LOCO_NUM-1)]);
		Serial.print("<");
		Serial.print(commandString);
		Serial.println(">");
		parse(commandString);
		F0_F4_OLD[(LOCO_NUM-1)] = F0_F4[(LOCO_NUM-1)];
      }

/////////////////////////////////////////////////////////////////////////////////////////////////////////              
//////////////////////////////  F5, F8 numbrds, vol, mute Functions  ////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////  

    SW910_AN_VAL = analogRead(SW910);  //Reading SW9, SW10 on A3
    if (SW910_AN_VAL > 401 and SW910_AN_VAL < 600) SW10 = 1;
    else SW10 = 0;
    if (SW10 == 1 and SW7 == 1) F5_F8[(LOCO_NUM-1)] = 184;  //SW7 (shift) and SW10 pressed for momentary F8 for volume change on Paragon2
    else F5_F8[(LOCO_NUM-1)] = 176;
      //writing 176 sets F5-F8 = 0
      //writing 184 sets F8 = 1 and F5-F7 = 0 
      //adding 1 to either sets F5 = 1
    if (SW10 == 1 and SW7 == 0) { //only SW10 (4.7K) "VOL/MUTE" pushed for latching/unlatching F8.
      F8_TOGGLE[(LOCO_NUM-1)] = !F8_TOGGLE[(LOCO_NUM-1)];
      delay (50000);
     }
    if (F8_TOGGLE[(LOCO_NUM-1)] == 1) F5_F8[(LOCO_NUM-1)] = 184;
    if (dccmanuf[(LOCO_NUM-1)] == 2) F5_F8[(LOCO_NUM-1)] ++; //For LokSound Select, turn on F5 for diesel numberboard lights by adding 1
    if (F5_F8[(LOCO_NUM-1)] != F5_F8_OLD[(LOCO_NUM-1)]){
        sprintf(commandString,"f %d %d", LOCO_NUM,F5_F8[(LOCO_NUM-1)]);
        Serial.print("<");
        Serial.print(commandString);
        Serial.println(">");
        parse(commandString);
        F5_F8_OLD[(LOCO_NUM-1)] = F5_F8[(LOCO_NUM-1)];
      }

/////////////////////////////////////////////////////////////////////////////////////////////////////////              
/////////////////////////////////////  F9-F12 Functions  ///////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////              

  if (SW1_4_AN_VAL > 401 and SW1_4_AN_VAL < 600) SW2 = 1; //F9, F10 sesnse (SW2)
	else SW2 = 0;
  
  if (SW2 == 1 and SW7 == 0) F9_F12 = 161;//SW2 on A2 pressed and SW7 (shift) not for F9 
	else if (SW2 == 1 and SW7 == 1) F9_F12 = 162;//SW2 on A2 pressed and SW7 (shift) pressed for F10 
	else if (SW2 == 0) F9_F12 = 160;
		  //writing 160 sets F9-F12 = 0
		  //writing 161 sets F9 = 1 and F10-F12 = 0,  F9 is shutdown sequence on Paragon, PM notch Up on LOKSOUND
		  //writing 162 sets F10 = 1 and F9, F11, F12 = 0,  F10 is coal shovel on Paragon, PM notch Down on LOKSOUND
    if (F9_F12 != F9_F12_OLD) {
      sprintf(commandString,"f %d %d", LOCO_NUM,F9_F12);
      Serial.print("<");
      Serial.print(commandString);
      Serial.println(">");
      parse(commandString);
      F9_F12_OLD = F9_F12;
      }

/////////////////////////////////////////////////////////////////////////////////////////////////////////              
/////////////////////////////////////  F13-F20 Sound Functions  ///////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////              

  if (SW910_AN_VAL > 601 and SW910_AN_VAL < 800) SW9 = 1; //F13-F20... sesnse (SW10)
	else SW9 = 0;
  
  if (SW9 == 1 and SW7 == 0) F13_F20 = 1;//SW9 on A2 pressed and SW7 (shift) not for F13
	else if (SW9 == 1 and SW7 == 1) F13_F20 = 2;//SW9 on A2 pressed and SW7 (shift) pressed for F15
	else F13_F20 = 0;
		  //writing 0 sets F13-F20 = 0
		  //writing 1 sets F13 = 1 and F14-F30 = 0, 
		  //writing 2 sets F14 = 1 and F13, F15-F20 = 0
      //writing 4 sets F15 = 1 and F13-F14, F16-F20 = 0
  if (F13_F20 != F13_F20_OLD) {
      sprintf(commandString,"f %d %d %d", LOCO_NUM, 222, F13_F20);
      Serial.print("<");
      Serial.print(commandString);
      Serial.println(">");
      parse(commandString);
      F13_F20_OLD = F13_F20;
      }

/////////////////////////////////////////////////////////////////////////////////////////////////////////              
/////////////////////////////////////  Throttle Loop Section  ///////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////  
     
   if (TRK_PWR_TOGGLE_PB == 1) {
    
    for (int i=0; i<126; i++) {  //Loop to capture change in throttle and do single write to loco
    
    SW78_AN_VAL = analogRead(SW78);  //Reading Throttle Multiplier switch on A4 
    if (SW78_AN_VAL > 451 and SW78_AN_VAL < 600) THR_MULT_PB = 150; //only 4.7K sw (FINE/THR_MULT) pushed
    else THR_MULT_PB = 1;
           
    SW1_4_AN_VAL = analogRead(SW1_4); //Reading up/dn throttle swithces on LCD SHIELD
    if (SW1_4_AN_VAL > 80 and SW1_4_AN_VAL < 200 and !(SW78_AN_VAL > 601 and SW78_AN_VAL < 800)) THROTTLE_UP_PB = 1;
    else THROTTLE_UP_PB = 0; //if throttle up is pushed and we are not selecting loco number change
    if (SW1_4_AN_VAL > 210 and SW1_4_AN_VAL < 400 and !(SW78_AN_VAL > 601 and SW78_AN_VAL < 800)) THROTTLE_DN_PB = 1;
    else THROTTLE_DN_PB = 0; //if throttle down is pushed and we are not selecting loco number change
    if (THROTTLE_UP_PB == 1) setpoint[(LOCO_NUM-1)] = (setpoint[(LOCO_NUM-1)] + 1);
    if (THROTTLE_DN_PB == 1) setpoint[(LOCO_NUM-1)] = (setpoint[(LOCO_NUM-1)] - 1);
    if (THROTTLE_UP_PB ==0 and THROTTLE_DN_PB == 0) i = 130;
    delay ((THR_MULT_PB * 200));
    if (setpoint[(LOCO_NUM-1)] > 125) setpoint[(LOCO_NUM-1)] = 126;
    if (setpoint[(LOCO_NUM-1)] < 0.75) setpoint[(LOCO_NUM-1)] = 0;
    if (setpointold[(LOCO_NUM-1)] != setpoint[(LOCO_NUM-1)]){
      lcd.setCursor(4, 1);
      lcd.print ("SPD");
      lcd.setCursor(8, 1);
      lcd.print ("   ");
      lcd.setCursor(8, 1);  
      lcd.print(setpoint[(LOCO_NUM-1)]);
      }      
    }
   }
    if (setpointold[(LOCO_NUM-1)] != setpoint[(LOCO_NUM-1)] and TRK_PWR_TOGGLE_PB == 1){
        sprintf(commandString,"t %d %d %d %d",LOCO_NUM,LOCO_NUM,setpoint[(LOCO_NUM-1)], FWD_REV_TOGGLE[(LOCO_NUM-1)]);
        parse(commandString);
        Serial.print("<");
        Serial.print(commandString);
        Serial.println(">");
        lcd.setCursor(4, 1);
        lcd.print ("SPD");
        lcd.setCursor(8, 1);
        lcd.print ("   ");
        lcd.setCursor(8, 1);  
        lcd.print(setpoint[(LOCO_NUM-1)]);
        setpointold[(LOCO_NUM-1)] = setpoint[(LOCO_NUM-1)];

      }
 
 
/////////////////////////////////////////////////////////////////////////////////////////////////////////           
/////////////////////////////////////////  END CJR MAIN MODS  ///////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////


  #elif COMM_TYPE == 1

    EthernetClient client=INTERFACE.available();

    if(client){
      while(client.connected() && client.available()){        // while there is data on the network
      c=client.read();
      if(c=='<')                    // start of new command
        sprintf(commandString,"");
      else if(c=='>')               // end of new command
        parse(commandString);                    
      else if(strlen(commandString)<MAX_COMMAND_LENGTH)    // if comandString still has space, append character just read from network
        sprintf(commandString,"%s%c",commandString,c);     // otherwise, character is ignored (but continue to look for '<' or '>')
      } // while
    }

  #endif

} // SerialCommand:process
   
///////////////////////////////////////////////////////////////////////////////

void SerialCommand::parse(char *com){
  
  switch(com[0]){

/***** SET ENGINE THROTTLES USING 128-STEP SPEED CONTROL ****/    

    case 't':       // <t REGISTER CAB SPEED DIRECTION>
/*
 *    sets the throttle for a given register/cab combination 
 *    
 *    REGISTER: an internal register number, from 1 through MAX_MAIN_REGISTERS (inclusive), to store the DCC packet used to control this throttle setting
 *    CAB:  the short (1-127) or long (128-10293) address of the engine decoder
 *    SPEED: throttle speed from 0-126, or -1 for emergency stop (resets SPEED to 0)
 *    DIRECTION: 1=forward, 0=reverse.  Setting direction when speed=0 or speed=-1 only effects directionality of cab lighting for a stopped train
 *    
 *    returns: <T REGISTER SPEED DIRECTION>
 *    
 */
      //Serial.println("throttle parse"); debug
      mRegs->setThrottle(com+1);
      break;

/***** OPERATE ENGINE DECODER FUNCTIONS F0-F28 ****/    

    case 'f':       // <f CAB BYTE1 [BYTE2]>
/*
 *    turns on and off engine decoder functions F0-F28 (F0 is sometimes called FL)  
 *    NOTE: setting requests transmitted directly to mobile engine decoder --- current state of engine functions is not stored by this program
 *    
 *    CAB:  the short (1-127) or long (128-10293) address of the engine decoder
 *    
 *    To set functions F0-F4 on (=1) or off (=0):
 *      
 *    BYTE1:  128 + F1*1 + F2*2 + F3*4 + F4*8 + F0*16
 *    BYTE2:  omitted
 *   
 *    To set functions F5-F8 on (=1) or off (=0):
 *   
 *    BYTE1:  176 + F5*1 + F6*2 + F7*4 + F8*8
 *    BYTE2:  omitted
 *   
 *    To set functions F9-F12 on (=1) or off (=0):
 *   
 *    BYTE1:  160 + F9*1 +F10*2 + F11*4 + F12*8
 *    BYTE2:  omitted
 *   
 *    To set functions F13-F20 on (=1) or off (=0):
 *   
 *    BYTE1: 222 
 *    BYTE2: F13*1 + F14*2 + F15*4 + F16*8 + F17*16 + F18*32 + F19*64 + F20*128
 *   
 *    To set functions F21-F28 on (=1) of off (=0):
 *   
 *    BYTE1: 223
 *    BYTE2: F21*1 + F22*2 + F23*4 + F24*8 + F25*16 + F26*32 + F27*64 + F28*128
 *   
 *    returns: NONE
 * 
 */
      mRegs->setFunction(com+1);

      break;
      
/***** OPERATE STATIONARY ACCESSORY DECODERS  ****/    

    case 'a':       // <a ADDRESS SUBADDRESS ACTIVATE>
/*
 *    turns an accessory (stationary) decoder on or off
 *    
 *    ADDRESS:  the primary address of the decoder (0-511)
 *    SUBADDRESS: the subaddress of the decoder (0-3)
 *    ACTIVATE: 1=on (set), 0=off (clear)
 *    
 *    Note that many decoders and controllers combine the ADDRESS and SUBADDRESS into a single number, N,
 *    from  1 through a max of 2044, where
 *    
 *    N = (ADDRESS - 1) * 4 + SUBADDRESS + 1, for all ADDRESS>0
 *    
 *    OR
 *    
 *    ADDRESS = INT((N - 1) / 4) + 1
 *    SUBADDRESS = (N - 1) % 4
 *    
 *    returns: NONE
 */
      mRegs->setAccessory(com+1);
      break;

/***** CREATE/EDIT/REMOVE/SHOW & OPERATE A TURN-OUT  ****/    

    case 'T':       // <T ID THROW>
/*
 *   <T ID THROW>:                sets turnout ID to either the "thrown" or "unthrown" position
 *   
 *   ID: the numeric ID (0-32767) of the turnout to control
 *   THROW: 0 (unthrown) or 1 (thrown)
 *   
 *   returns: <H ID THROW> or <X> if turnout ID does not exist
 *   
 *   *** SEE ACCESSORIES.CPP FOR COMPLETE INFO ON THE DIFFERENT VARIATIONS OF THE "T" COMMAND
 *   USED TO CREATE/EDIT/REMOVE/SHOW TURNOUT DEFINITIONS
 */
      Turnout::parse(com+1);
      break;

/***** CREATE/EDIT/REMOVE/SHOW & OPERATE AN OUTPUT PIN  ****/    

    case 'Z':       // <Z ID ACTIVATE>
/*
 *   <Z ID ACTIVATE>:          sets output ID to either the "active" or "inactive" state
 *   
 *   ID: the numeric ID (0-32767) of the output to control
 *   ACTIVATE: 0 (active) or 1 (inactive)
 *   
 *   returns: <Y ID ACTIVATE> or <X> if output ID does not exist
 *   
 *   *** SEE OUTPUTS.CPP FOR COMPLETE INFO ON THE DIFFERENT VARIATIONS OF THE "O" COMMAND
 *   USED TO CREATE/EDIT/REMOVE/SHOW TURNOUT DEFINITIONS
 */
      Output::parse(com+1);
      break;
      
/***** CREATE/EDIT/REMOVE/SHOW A SENSOR  ****/    

    case 'S': 
/*   
 *   *** SEE SENSOR.CPP FOR COMPLETE INFO ON THE DIFFERENT VARIATIONS OF THE "S" COMMAND
 *   USED TO CREATE/EDIT/REMOVE/SHOW SENSOR DEFINITIONS
 */
      Sensor::parse(com+1);
      break;

/***** SHOW STATUS OF ALL SENSORS ****/

    case 'Q':         // <Q>
/*
 *    returns: the status of each sensor ID in the form <Q ID> (active) or <q ID> (not active)
 */
      Sensor::status();
      break;

/***** WRITE CONFIGURATION VARIABLE BYTE TO ENGINE DECODER ON MAIN OPERATIONS TRACK  ****/    

    case 'w':      // <w CAB CV VALUE>
/*
 *    writes, without any verification, a Configuration Variable to the decoder of an engine on the main operations track
 *    
 *    CAB:  the short (1-127) or long (128-10293) address of the engine decoder 
 *    CV: the number of the Configuration Variable memory location in the decoder to write to (1-1024)
 *    VALUE: the value to be written to the Configuration Variable memory location (0-255)
 *    
 *    returns: NONE
*/    
      mRegs->writeCVByteMain(com+1);
      break;      

/***** WRITE CONFIGURATION VARIABLE BIT TO ENGINE DECODER ON MAIN OPERATIONS TRACK  ****/    

    case 'b':      // <b CAB CV BIT VALUE>
/*
 *    writes, without any verification, a single bit within a Configuration Variable to the decoder of an engine on the main operations track
 *    
 *    CAB:  the short (1-127) or long (128-10293) address of the engine decoder 
 *    CV: the number of the Configuration Variable memory location in the decoder to write to (1-1024)
 *    BIT: the bit number of the Configurarion Variable regsiter to write (0-7)
 *    VALUE: the value of the bit to be written (0-1)
 *    
 *    returns: NONE
*/        
      mRegs->writeCVBitMain(com+1);
      break;      

/***** WRITE CONFIGURATION VARIABLE BYTE TO ENGINE DECODER ON PROGRAMMING TRACK  ****/    

    case 'W':      // <W CV VALUE CALLBACKNUM CALLBACKSUB>
/*
 *    writes, and then verifies, a Configuration Variable to the decoder of an engine on the programming track
 *    
 *    CV: the number of the Configuration Variable memory location in the decoder to write to (1-1024)
 *    VALUE: the value to be written to the Configuration Variable memory location (0-255) 
 *    CALLBACKNUM: an arbitrary integer (0-32767) that is ignored by the Base Station and is simply echoed back in the output - useful for external programs that call this function
 *    CALLBACKSUB: a second arbitrary integer (0-32767) that is ignored by the Base Station and is simply echoed back in the output - useful for external programs (e.g. DCC++ Interface) that call this function
 *    
 *    returns: <r CALLBACKNUM|CALLBACKSUB|CV Value)
 *    where VALUE is a number from 0-255 as read from the requested CV, or -1 if verificaiton read fails
*/    
      pRegs->writeCVByte(com+1);
      break;      

/***** WRITE CONFIGURATION VARIABLE BIT TO ENGINE DECODER ON PROGRAMMING TRACK  ****/    

    case 'B':      // <B CV BIT VALUE CALLBACKNUM CALLBACKSUB>
/*
 *    writes, and then verifies, a single bit within a Configuration Variable to the decoder of an engine on the programming track
 *    
 *    CV: the number of the Configuration Variable memory location in the decoder to write to (1-1024)
 *    BIT: the bit number of the Configurarion Variable memory location to write (0-7)
 *    VALUE: the value of the bit to be written (0-1)
 *    CALLBACKNUM: an arbitrary integer (0-32767) that is ignored by the Base Station and is simply echoed back in the output - useful for external programs that call this function
 *    CALLBACKSUB: a second arbitrary integer (0-32767) that is ignored by the Base Station and is simply echoed back in the output - useful for external programs (e.g. DCC++ Interface) that call this function
 *    
 *    returns: <r CALLBACKNUM|CALLBACKSUB|CV BIT VALUE)
 *    where VALUE is a number from 0-1 as read from the requested CV bit, or -1 if verificaiton read fails
*/    
      pRegs->writeCVBit(com+1);
      break;      

/***** READ CONFIGURATION VARIABLE BYTE FROM ENGINE DECODER ON PROGRAMMING TRACK  ****/    

    case 'R':     // <R CV CALLBACKNUM CALLBACKSUB>
/*    
 *    reads a Configuration Variable from the decoder of an engine on the programming track
 *    
 *    CV: the number of the Configuration Variable memory location in the decoder to read from (1-1024)
 *    CALLBACKNUM: an arbitrary integer (0-32767) that is ignored by the Base Station and is simply echoed back in the output - useful for external programs that call this function
 *    CALLBACKSUB: a second arbitrary integer (0-32767) that is ignored by the Base Station and is simply echoed back in the output - useful for external programs (e.g. DCC++ Interface) that call this function
 *    
 *    returns: <r CALLBACKNUM|CALLBACKSUB|CV VALUE)
 *    where VALUE is a number from 0-255 as read from the requested CV, or -1 if read could not be verified
*/    
      pRegs->readCV(com+1);
      break;

/////////////////////////////////////////////////////////////////////////////////////////////////////////           
//////////////////////////////////  CJR MOD- ADDING PROG MODE  //////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////    

 /***** TURN ON POWER FROM MOTOR SHIELD TO TRACKS  ****/   

    case '2':      // <2>
/*   
 *    enables power from the motor shield to the programming track, only done via serial
 *    
 *    returns: <p2>
 */    
     for (int n=0; n<LOCOS; n++) { //resets all locos to forward, speed = 0 whenever TRK_PWR_TOGGLE_PB changes state
        setpoint[n] = 0;
        FWD_REV_TOGGLE[n] = 1;
        sprintf(commandString,"t %d %d %d %d",(n+1),(n+1),setpoint[(n)], FWD_REV_TOGGLE[(n)]);
        parse(commandString);
        Serial.print("<");
        Serial.print(commandString);
        Serial.println(">");
        setpointold[n] = 0;
     }
     digitalWrite(SIGNAL_ENABLE_PIN_PROG,HIGH); //turns off main track, turns on prog track. 
     digitalWrite(SIGNAL_ENABLE_PIN_MAIN,LOW);
     PROG_MODE = 1;
     INTERFACE.print("<p2>");
     lcd.setCursor(0, 0);
     lcd.print ("PROGRAM MODE     ");
     TRK_PWR_TOGGLE_PB = 0;
     lcd.setCursor(0, 1);  
     lcd.print("                 "); //blank display second line when in prog mode.
     break;

    case '1':      // <1>
/*   
 *    enables power from the motor shield to the main operations track
 *    
 *    returns: <p1>
 */    
     digitalWrite(SIGNAL_ENABLE_PIN_PROG,LOW);
     digitalWrite(SIGNAL_ENABLE_PIN_MAIN,HIGH);
     delay (10000);
     INTERFACE.print("<p1>");
     PROG_MODE = 0;
     TRK_PWR_TOGGLE_PB = 1; 
     lcd.setCursor(13, 1);  
     lcd.print("ON ");
     lcd.setCursor(0, 0);
     lcd.print ("Locomotive: ");
     lcd.setCursor (12, 0);
     lcd.print ("  ");
     lcd.setCursor (12, 0);
     lcd.print (LOCO_NUM);     
     break;
          
/***** TURN OFF POWER FROM MOTOR SHIELD TO TRACKS  ****/    

    case '0':     // <0>
/*   
 *    disables power from the motor shield to the main operations and programming tracks
 *    
 *    returns: <p0>
 */
     digitalWrite(SIGNAL_ENABLE_PIN_PROG,LOW);
     digitalWrite(SIGNAL_ENABLE_PIN_MAIN,LOW);
     delay (10000);
     INTERFACE.print("<p0>");
     PROG_MODE = 0;
     TRK_PWR_TOGGLE_PB = 0;
     lcd.setCursor(13, 1);  
     lcd.print("OFF");
     lcd.setCursor(8, 1);  
     lcd.print("   "); //when track power is turned off, blank out speed on LCD.
     lcd.setCursor(0, 0);
     lcd.print ("Locomotive:     ");
     break;
	 
/////////////////////////////////////////////////////////////////////////////////////////////////////////           
////////////////////////////////////////  END CJR MODS  /////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////     

/***** READ MAIN OPERATIONS TRACK CURRENT  ****/    

    case 'c':     // <c>
/*
 *    reads current being drawn on main operations track
 *    
 *    returns: <a CURRENT> 
 *    where CURRENT = 0-1024, based on exponentially-smoothed weighting scheme
 */
      INTERFACE.print("<a");
      INTERFACE.print(int(mMonitor->current));
      INTERFACE.print(">");
      break;

/***** READ STATUS OF DCC++ BASE STATION  ****/    

    case 's':      // <s>
/*
 *    returns status messages containing track power status, throttle status, turn-out status, and a version number
 *    NOTE: this is very useful as a first command for an interface to send to this sketch in order to verify connectivity and update any GUI to reflect actual throttle and turn-out settings
 *    
 *    returns: series of status messages that can be read by an interface to determine status of DCC++ Base Station and important settings
 */
      if(digitalRead(SIGNAL_ENABLE_PIN_MAIN)==LOW)      // could check either PROG or MAIN
        INTERFACE.print("<p0>");
      else
        INTERFACE.print("<p1>");

      for(int i=1;i<=MAX_MAIN_REGISTERS;i++){
        if(mRegs->speedTable[i]==0)
          continue;
        INTERFACE.print("<T");
        INTERFACE.print(i); INTERFACE.print(" ");
        if(mRegs->speedTable[i]>0){
          INTERFACE.print(mRegs->speedTable[i]);
          INTERFACE.print(" 1>");
        } else{
          INTERFACE.print(-mRegs->speedTable[i]);
          INTERFACE.print(" 0>");
        }          
      }
      INTERFACE.print("<iDCC++ BASE STATION FOR ARDUINO ");
      INTERFACE.print(ARDUINO_TYPE);
      INTERFACE.print(" / ");
      INTERFACE.print(MOTOR_SHIELD_NAME);
      INTERFACE.print(": V-");
      INTERFACE.print(VERSION);
      INTERFACE.print(" / ");
      INTERFACE.print(__DATE__);
      INTERFACE.print(" ");
      INTERFACE.print(__TIME__);
      INTERFACE.print(">");

      INTERFACE.print("<N");
      INTERFACE.print(COMM_TYPE);
      INTERFACE.print(": ");

      #if COMM_TYPE == 0
        INTERFACE.print("SERIAL>");
      #elif COMM_TYPE == 1
        INTERFACE.print(Ethernet.localIP());
        INTERFACE.print(">");
      #endif
      
      Turnout::show();
      Output::show();
                        
      break;

/***** STORE SETTINGS IN EEPROM  ****/    

    case 'E':     // <E>
/*
 *    stores settings for turnouts and sensors EEPROM
 *    
 *    returns: <e nTurnouts nSensors>
*/
     
    EEStore::store();
    INTERFACE.print("<e ");
    INTERFACE.print(EEStore::eeStore->data.nTurnouts);
    INTERFACE.print(" ");
    INTERFACE.print(EEStore::eeStore->data.nSensors);
    INTERFACE.print(" ");
    INTERFACE.print(EEStore::eeStore->data.nOutputs);
    INTERFACE.print(">");
    break;
    
/***** CLEAR SETTINGS IN EEPROM  ****/    

    case 'e':     // <e>
/*
 *    clears settings for Turnouts in EEPROM
 *    
 *    returns: <O>
*/
     
    EEStore::clear();
    INTERFACE.print("<O>");
    break;

/***** PRINT CARRIAGE RETURN IN SERIAL MONITOR WINDOW  ****/    
                
    case ' ':     // < >                
/*
 *    simply prints a carriage return - useful when interacting with Ardiuno through serial monitor window
 *    
 *    returns: a carriage return
*/
      INTERFACE.println("");
      break;  

///          
/// THE FOLLOWING COMMANDS ARE NOT NEEDED FOR NORMAL OPERATIONS AND ARE ONLY USED FOR TESTING AND DEBUGGING PURPOSES
/// PLEASE SEE SPECIFIC WARNINGS IN EACH COMMAND BELOW
///

/***** ENTER DIAGNOSTIC MODE  ****/    

    case 'D':       // <D>  
/*
 *    changes the clock speed of the chip and the pre-scaler for the timers so that you can visually see the DCC signals flickering with an LED
 *    SERIAL COMMUNICAITON WILL BE INTERUPTED ONCE THIS COMMAND IS ISSUED - MUST RESET BOARD OR RE-OPEN SERIAL WINDOW TO RE-ESTABLISH COMMS
 */

    Serial.println("\nEntering Diagnostic Mode...");
    delay(1000);
    
    bitClear(TCCR1B,CS12);    // set Timer 1 prescale=8 - SLOWS NORMAL SPEED BY FACTOR OF 8
    bitSet(TCCR1B,CS11);
    bitClear(TCCR1B,CS10);

    #ifdef ARDUINO_AVR_UNO      // Configuration for UNO

      bitSet(TCCR0B,CS02);    // set Timer 0 prescale=256 - SLOWS NORMAL SPEED BY A FACTOR OF 4
      bitClear(TCCR0B,CS01);
      bitClear(TCCR0B,CS00);
      
    #else                     // Configuration for MEGA

      bitClear(TCCR3B,CS32);    // set Timer 3 prescale=8 - SLOWS NORMAL SPEED BY A FACTOR OF 8
      bitSet(TCCR3B,CS31);
      bitClear(TCCR3B,CS30);

    #endif

    CLKPR=0x80;           // THIS SLOWS DOWN SYSYEM CLOCK BY FACTOR OF 256
    CLKPR=0x08;           // BOARD MUST BE RESET TO RESUME NORMAL OPERATIONS

    break;

/***** WRITE A DCC PACKET TO ONE OF THE REGSITERS DRIVING THE MAIN OPERATIONS TRACK  ****/    
      
    case 'M':       // <M REGISTER BYTE1 BYTE2 [BYTE3] [BYTE4] [BYTE5]>
/*
 *   writes a DCC packet of two, three, four, or five hexidecimal bytes to a register driving the main operations track
 *   FOR DEBUGGING AND TESTING PURPOSES ONLY.  DO NOT USE UNLESS YOU KNOW HOW TO CONSTRUCT NMRA DCC PACKETS - YOU CAN INADVERTENTLY RE-PROGRAM YOUR ENGINE DECODER
 *   
 *    REGISTER: an internal register number, from 0 through MAX_MAIN_REGISTERS (inclusive), to write (if REGISTER=0) or write and store (if REGISTER>0) the packet 
 *    BYTE1:  first hexidecimal byte in the packet
 *    BYTE2:  second hexidecimal byte in the packet
 *    BYTE3:  optional third hexidecimal byte in the packet
 *    BYTE4:  optional fourth hexidecimal byte in the packet
 *    BYTE5:  optional fifth hexidecimal byte in the packet
 *   
 *    returns: NONE   
 */
      mRegs->writeTextPacket(com+1);
      break;

/***** WRITE A DCC PACKET TO ONE OF THE REGSITERS DRIVING THE MAIN OPERATIONS TRACK  ****/    

    case 'P':       // <P REGISTER BYTE1 BYTE2 [BYTE3] [BYTE4] [BYTE5]>
/*
 *   writes a DCC packet of two, three, four, or five hexidecimal bytes to a register driving the programming track
 *   FOR DEBUGGING AND TESTING PURPOSES ONLY.  DO NOT USE UNLESS YOU KNOW HOW TO CONSTRUCT NMRA DCC PACKETS - YOU CAN INADVERTENTLY RE-PROGRAM YOUR ENGINE DECODER
 *   
 *    REGISTER: an internal register number, from 0 through MAX_MAIN_REGISTERS (inclusive), to write (if REGISTER=0) or write and store (if REGISTER>0) the packet 
 *    BYTE1:  first hexidecimal byte in the packet
 *    BYTE2:  second hexidecimal byte in the packet
 *    BYTE3:  optional third hexidecimal byte in the packet
 *    BYTE4:  optional fourth hexidecimal byte in the packet
 *    BYTE5:  optional fifth hexidecimal byte in the packet
 *   
 *    returns: NONE   
 */
      pRegs->writeTextPacket(com+1);
      break;
            
/***** ATTEMPTS TO DETERMINE HOW MUCH FREE SRAM IS AVAILABLE IN ARDUINO  ****/        
      
    case 'F':     // <F>
/*
 *     measure amount of free SRAM memory left on the Arduino based on trick found on the internet.
 *     Useful when setting dynamic array sizes, considering the Uno only has 2048 bytes of dynamic SRAM.
 *     Unfortunately not very reliable --- would be great to find a better method
 *     
 *     returns: <f MEM>
 *     where MEM is the number of free bytes remaining in the Arduino's SRAM
 */
      int v; 
      INTERFACE.print("<f");
      INTERFACE.print((int) &v - (__brkval == 0 ? (int) &__heap_start : (int) __brkval));
      INTERFACE.print(">");
      break;

/***** LISTS BIT CONTENTS OF ALL INTERNAL DCC PACKET REGISTERS  ****/        

    case 'L':     // <L>
/*
 *    lists the packet contents of the main operations track registers and the programming track registers
 *    FOR DIAGNOSTIC AND TESTING USE ONLY
 */
      INTERFACE.println("");
      for(Register *p=mRegs->reg;p<=mRegs->maxLoadedReg;p++){
        INTERFACE.print("M"); INTERFACE.print((int)(p-mRegs->reg)); INTERFACE.print(":\t");
        INTERFACE.print((int)p); INTERFACE.print("\t");
        INTERFACE.print((int)p->activePacket); INTERFACE.print("\t");
        INTERFACE.print(p->activePacket->nBits); INTERFACE.print("\t");
        for(int i=0;i<10;i++){
          INTERFACE.print(p->activePacket->buf[i],HEX); INTERFACE.print("\t");
        }
        INTERFACE.println("");
      }
      for(Register *p=pRegs->reg;p<=pRegs->maxLoadedReg;p++){
        INTERFACE.print("P"); INTERFACE.print((int)(p-pRegs->reg)); INTERFACE.print(":\t");
        INTERFACE.print((int)p); INTERFACE.print("\t");
        INTERFACE.print((int)p->activePacket); INTERFACE.print("\t");
        INTERFACE.print(p->activePacket->nBits); INTERFACE.print("\t");
        for(int i=0;i<10;i++){
          INTERFACE.print(p->activePacket->buf[i],HEX); INTERFACE.print("\t");
        }
        INTERFACE.println("");
      }
      INTERFACE.println("");
      break;

  } // switch
}; // SerialCommand::parse

///////////////////////////////////////////////////////////////////////////////


